<?php

namespace App\Controllers;

use App\Models\UsuarioModel;

class Auth extends BaseController
{
    protected $usuarioModel;

    public function __construct()
    {
        $this->usuarioModel = model(UsuarioModel::class);
        helper(['form', 'url', 'email']);
    }

    public function login()
    {
        // Si ya esta logueado, redirigir segun su rol directamente
        if (session()->get('isLoggedIn')) {
            return $this->redirectByRole(session()->get('rol'));
        }
        
        if ($this->request->is('post')) {
            $email = trim($this->request->getPost('email'));
            $password = $this->request->getPost('password');
            
            $user = $this->usuarioModel->login($email, $password);
            
            if ($user) {
                // Seteamos la sesion usando tu metodo privado
                $this->setUserSession($user);
                
                // Redireccionamos segun el rol del usuario que acaba de entrar
                return $this->redirectByRole($user['rol']);
            }
            
            return redirect()->back()->withInput()->with('error', 'Credenciales incorrectas.');
        }
        
        return view('auth/login');
    }
    
        public function register() 
{
    // 1. Impedir registro si ya está logueado
    if (session()->get('isLoggedIn')) {
        return $this->redirectByRole(session()->get('rol'));
    }

    if ($this->request->is('post')) {
        
        // 2. Seguridad Anti-Bot (Honeypot)
        if (!empty($this->request->getPost('hp_field'))) {
            return redirect()->to(base_url('/'))->with('error', 'Actividad sospechosa.');
        }

        // 3. Reglas de Validación
        $rules = [
            'nombre'       => 'required|min_length[3]|max_length[50]',
            'email'        => 'required|valid_email|is_unique[usuarios.email]',
            'password'     => 'required|min_length[6]',
            'pass_confirm' => 'required|matches[password]',
            'rol'          => 'required|in_list[modelo,usuario]'
        ];

        if (!$this->validate($rules)) {
            return redirect()->back()->withInput()->with('error', $this->validator->listErrors());
        }

        // 4. Preparación de Datos
        $rol = $this->request->getPost('rol');
        $data = [
            'nombre'   => trim($this->request->getPost('nombre')),
            'email'    => strtolower(trim($this->request->getPost('email'))),
            'password' => password_hash($this->request->getPost('password'), PASSWORD_DEFAULT),
            'rol'      => $rol,
            'activo'   => 1
        ];

        // 5. Inserción y Notificación
        try {
            if ($this->usuarioModel->insert($data)) {
                
                helper('email');
                $emailData = ['nombre' => $data['nombre'], 'email' => $data['email'], 'rol' => $data['rol']];

                // Notificaciones (silenciosas para no interrumpir el flujo)
                try {
                    enviar_notificacion($data['email'], '¡Bienvenido a Calientitas! 🔥', 'bienvenida_usuario', $emailData);
                    enviar_notificacion('hola@kmarketexpress.cl', 'NUEVO REGISTRO: ' . strtoupper($rol), 'admin_nuevo_usuario', $emailData);
                } catch (\Exception $e) {
                    log_message('error', 'Error en envío de correos: ' . $e->getMessage());
                }

                $msg = ($rol === 'modelo') ? '¡Cuenta creada! Inicia sesión para configurar tu perfil.' : '¡Registro exitoso! Ya puedes iniciar sesión.';
                return redirect()->to(base_url('login'))->with('success', $msg);

            } else {
                return redirect()->back()->withInput()->with('error', 'No se pudo crear la cuenta.');
            }
        } catch (\Exception $e) {
            return redirect()->back()->withInput()->with('error', 'Error de sistema: ' . $e->getMessage());
        }
    }

    return view('auth/registro');
}

    /**
     * Logica de redireccion segun el perfil del usuario
     */
    private function redirectByRole($rol)
    {
        switch ($rol) {
            case 'admin':
                return redirect()->to(base_url('admin/dashboard'));
            case 'modelo':
                // Redirigir a su panel de control para que gestione su ficha
                return redirect()->to(base_url('modelo/dashboard'));
            case 'usuario':
                // CAMBIO: En lugar de enviarlo a /usuario/panel, lo mandamos al Home
                // Esto permite que el usuario empiece a navegar y ver modelos de inmediato.
                return redirect()->to(base_url('/'))->with('success', '¡Bienvenido de nuevo!');
            default:
                return redirect()->to(base_url('/'));
        }
    }

    private function setUserSession($user)
    {
        $data = [
            'user_id'    => $user['id'],
            'email'      => $user['email'],
            'rol'        => $user['rol'],
            'nombre'     => $user['nombre'],
            'isLoggedIn' => true,
        ];

        session()->set($data);
        return true;
    }

    public function logout()
    {
        // 1. Obtenemos los datos antes de destruir la sesión
        $userId = session()->get('user_id');
        $rol    = session()->get('rol');
    
        // 2. Si es una modelo, cambiamos su estado a "No Disponible" (0)
        if ($rol === 'modelo' && $userId) {
            $db = \Config\Database::connect();
            $db->table('modelos')
               ->where('usuario_id', $userId)
               ->update(['disponible_ahora' => 0]);
        }
    
        // 3. Ahora sí, destruimos la sesión y redirigimos
        session()->destroy();
        return redirect()->to(base_url('login'))->with('success', 'Sesión cerrada. Tu estado ahora es: No Disponible.');
    }
    public function recuperar() {
        return view('auth/recuperar_password');
    }

    public function enviarToken() {
    $email = $this->request->getPost('email');
    
    // Validación básica de formato
    if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
        return redirect()->back()->with('error', 'Por favor, ingresa un correo electrónico válido.');
    }

    $usuarioModel = new \App\Models\UsuarioModel();
    $user = $usuarioModel->where('email', $email)->first();

    if ($user) {
        $token = bin2hex(random_bytes(20));
        $expira = date('Y-m-d H:i:s', strtotime('+1 hour'));

        $db = \Config\Database::connect();
        $db->table('recuperacion_password')->insert([
            'email' => $email,
            'token' => $token,
            'expira_en' => $expira
        ]);

        try {
            helper('email');
            $enlace = base_url("auth/restablecer/$token");
            enviar_notificacion($email, 'Restablecer contraseña - Calientitas', 'password_reset', [
                'nombre' => $user['nombre'],
                'enlace' => $enlace
            ]);
        } catch (\Exception $e) {
            return redirect()->back()->with('error', 'Hubo un problema al enviar el correo. Inténtalo más tarde.');
        }
    }

    // Mensaje de éxito genérico (por seguridad)
    return redirect()->back()->with('success', '¡Enlace enviado! Revisa tu bandeja de entrada (y la carpeta de spam).');
}
    public function restablecer($token) {
        $db = \Config\Database::connect();
        $registro = $db->table('recuperacion_password')
                       ->where('token', $token)
                       ->where('expira_en >', date('Y-m-d H:i:s'))
                       ->get()->getRowArray();
    
        if (!$registro) {
            return redirect()->to(base_url('login'))->with('error', 'El enlace es inválido o ha expirado.');
        }
    
        return view('auth/restablecer_password', ['token' => $token]);
    }   

    public function actualizarPassword() {
        $token = $this->request->getPost('token');
        $password = $this->request->getPost('password');
    
        $db = \Config\Database::connect();
        $registro = $db->table('recuperacion_password')
                       ->where('token', $token)
                       ->where('expira_en >', date('Y-m-d H:i:s'))
                       ->get()->getRowArray();
    
        if (!$registro) {
            return redirect()->to(base_url('login'))->with('error', 'Sesión de recuperación expirada.');
        }
    
        // Actualizar la contraseña del usuario (asumiendo que usas password_hash)
        $usuarioModel = new \App\Models\UsuarioModel();
        $nuevoPasswordHash = password_hash($password, PASSWORD_DEFAULT);
        
        $usuarioModel->where('email', $registro['email'])
                     ->set(['password' => $nuevoPasswordHash])
                     ->update();
    
        // Borrar el token para que no se use de nuevo
        $db->table('recuperacion_password')->where('email', $registro['email'])->delete();
    
        return redirect()->to(base_url('login'))->with('success', 'Contraseña actualizada. Ya puedes iniciar sesión.');
    }
    
}